function [engine ] = REVS_build_engine( displacement, mapspeed, maptorque, mapfuel, WOTspeed, WOTtorque, CTspeed, CTtorque, varargin )
%engineMap2Alpha Summary of this function goes here
%   
% Inputs:    mapspeed, maptorque, and mapfuel are three vectors that define a
% set of scatter points in 3d space.
% 
% mapspeed - vector containing engine speed values to form the x-axis of
% the contour map. Units can be RPM or Rad/second
% 
% maptoque - vector containing engine torque values to form the y-axis of
% the contour map. Units can be Newton*meters or BMEP
% 
% mapfuel - vector containing the values for the z-axis of the contour map.
% (such as bsfc, efficiency, etc.)
% 
% WOTspeed - vector containing the engine speed values to define the Wide
% Open Throttle torque curve. Units can be RPM or Rad/s
% 
% WOTtorque - vector contatining the engine torque values to define the
% Wide Open Throttle torque curve. Units can be Nm or BMEP.
% 
% CTspeed - vector containing the speed values for Closed Throttle torque
% curve. Note: if not specified, the default value below will be used.
% 
% CTtorque - vector containing the torque values for the Closed Throttle
% torque curve. Note: if not specified, the default value below will be
% used.
% 
% Optional Parameters: (used as name, value pairs)
% CTspeed - defined above
% CTtorque - defined above
% 
% noplots - true or false(default): if true, enineMap2Alpha will not make a
% call to makeREVSplots.m.
% 


%% Handle WOT Curve
if isempty( WOTspeed )
	
	% Find WOT from speed and load boundary
    xscat = [max(mapspeed)*1.1;mapspeed;0];
    yscat = [-1;maptorque;0];

    k = convhull(xscat, yscat); 
	
    WOTspeed = xscat(k);
    WOTtorque = yscat(k);  
	
	WOTspeed = WOTspeed( WOTtorque >= 0 );
	WOTtorque = WOTtorque( WOTtorque >= 0 );
	
	
	
	
	[WOTspeed, sort_idx] = unique(WOTspeed);
	WOTtorque = WOTtorque(sort_idx);
	

end

if WOTtorque(1) ~= 0
	WOTspeed = [0;WOTspeed(:)];
	WOTtorque = [0;WOTtorque(:)];
elseif WOTspeed(1) ~= 0
	WOTspeed(1) = 0;
end


% 
if WOTtorque(end) > 0
	WOTspeed_new = [WOTspeed(:); WOTspeed(end) .* [1.05; 1.15] ];
	WOTtorque = interp1( WOTspeed(:), WOTtorque(:), WOTspeed_new, 'linear','extrap' );
	WOTspeed = WOTspeed_new;
	WOTtorque(end-1) = min( WOTtorque(end-1), WOTtorque(end-2 ) );		% Force downward slope
	WOTtorque(end) = min(WOTtorque(end), 0.0 );							% Force to Zero

end

	



%% Handle CTP Curve
if ~isempty( CTspeed )
% Closed Throttle Curve Provided - Good	
elseif sum( maptorque < 0 ) > 5
		
	xscat = [max(mapspeed);mapspeed;0];
    yscat = [1;maptorque;1];

    k = convhull(xscat, yscat); 
    CTspeed = xscat(k);
    CTtorque = yscat(k);  
	
	CTspeed = CTspeed( CTtorque < 0 );
	CTtorque = CTtorque( CTtorque < 0 );
	
	[CTspeed, sort_idx] = unique(CTspeed);
	CTtorque = CTtorque(sort_idx);
	
		
		
	else
	% Sscale default closed throttle curve
	CTspeed = convert.rpm2radps*[657.2 812.1 1050 1991 3008 4064 4555]; 
    CTtorque = [-29.57 -34.01 -36.12 -40.44 -47.98 -57.72 -62.77] * displacement / 2.4; 
end


	CTspeed_new = unique([0; CTspeed(:); WOTspeed(end)]);
	CTtorque = interp1( CTspeed, CTtorque, CTspeed_new,'linear','extrap');
	CTspeed = CTspeed_new;
	

%% Construct Class

engine = class_REVS_engine;

engine.fuel = class_REVS_fuel;
engine.displacement_L = displacement;

% Define Wide-Open Throttle (WOT) Torque Table
engine.full_throttle_speed_radps  = WOTspeed;
engine.full_throttle_torque_Nm = WOTtorque; 

% Define Closed Throttle Torque Table
engine.closed_throttle_speed_radps     = CTspeed; 
engine.closed_throttle_torque_Nm       = CTtorque;

% append Close Throttle values to map scatter points
mapspeed = [mapspeed; CTspeed];
maptorque = [maptorque; CTtorque];
fuel_gps = [mapfuel; zeros(size(CTtorque))];

% create linearized grid of the map, in an x and a y grid vector
% Xgv = round(linspace(0,max(WOTspeed),50));
% Ygv = round(linspace(1.1 * min(CTtorque), 1.05* max(WOTtorque), 50 ));



Xgv = find_clusters( mapspeed, 5);
Xgv(end+1) = 0;
Xgv(end+1) = max(WOTspeed);
Xgv = unique(Xgv);




Ygv = find_clusters( maptorque, 5);
Ygv(end+1) = 1.1 * min(CTtorque);
Ygv(end+1) = 1.05* max(WOTtorque);
Ygv = unique(Ygv);


% interpolate map scatter points and grid to generate a surface matrix
% Zgrid = scatter2surf(mapspeed, maptorque, fuel_gps, Xgv, Ygv,'method','gridfit','interp','bilinear','xnormalize','ynormalize','ysmooth',3,'xsmooth',3);
 Zgrid = scatter2surf(mapspeed, maptorque, fuel_gps, Xgv, Ygv,'method','gridfit','interp','bilinear','xnormalize','ynormalize','smoother','gradient','xsmooth',5,'ysmooth',10);
% Zgrid = scatter2surf(mapspeed, maptorque, fuel_gps, Xgv, Ygv,'method','scatterinterp','interp','natural','xnormalize','ynormalize');

engine.fuel_map_gps = max(Zgrid, 0); % change negative values to 0

% assign the grid vectors to the surface X and Y definitions.
engine.fuel_map_speed_radps = Xgv;
engine.fuel_map_torque_Nm = Ygv;




end


